﻿#
# Read current measured values example
#
# Requirements: Python 3.6.1 ( www.python.org )
#               Tx5xx/Tx6xx device with firmware version 1-5-7-4 or higher
#
#  Purpose of this example is demonstration of communication with Tx5xx/Tx6xx device.
#  It is not intended to be used in production environment without further changes.
#

#Connection parameters
TCP_IP = '192.168.1.213'
TCP_PORT = 502

import socket, struct

sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
sock.connect((TCP_IP, TCP_PORT))

try:
    req = struct.pack('12B', 0x00, 0x00, 0x00, 0x00, 0x00, 0x06, 0x01, 0x03, 0x00, 0x30, 0x00, 0x04)
    #send ModbusTCP request
    sock.send(req)
    '''
    Modbus TCP request:
       0x00 0x00 0x00 0x00 - transaction ID + protocol ID
       0x00 0x06           - length of following bytes
       0x01                - unit ID
       0x03                - function code (read holding registers)
       0x00 0x30           - register address (from manual - 0x0030=temperature)
       0x00 0x04           - number of registers for reading (four measured values)
    '''

    #read ModbusTCP response
    rcv = sock.recv(64)
    #decode response
    out = struct.unpack(">IHccchhhh", rcv)
    print("Temperature:\t\t", out[5]/10,
          "\nReleative humidity:\t", out[6]/10,
          "\nComputed quantity:\t", out[7]/10,
          "\nCO2 level:\t\t", out[8])
    '''
    Modbus TCP response:
       0x00 0x00 0x00 0x00 - transaction ID + protocol ID
       0x00 0x0B           - length of following bytes
       0x01                - unit ID
       0x03                - function code (read holding registers)
       0x08                - bytes count (=4x2)
       0x00 0xF3 0x01 0x58 0x00 0x4C 0x04 0xDF - response data
          0x00 0xF3 = 243 = 24.3°C (temperature)
          0x01 0x58 = 344 = 34.4%RH (relative humidity)
          0x00 0x4C = 76 = 7.6°C (dew point)
          0x04 0xDF = 1247 = 1247ppm (CO2)
    '''
    
finally:
    sock.close()
